/********************************************************

	Audio renderer object for file output
	Copyright 2001 Alexander Oelzant (aoe@mars.tuwien.ac.at)
	  derived from lib/aviplay/audio.cpp
	Copyright 2000 Eugene Kuznetsov  (divx@euro.ru)

*********************************************************/

#undef DEBUG_TOCONSOLE
#define QUIET

#ifndef QUIET
#define debug_out(X) Debug cout<<X<<": "<<(t2-t1)/freq<<" ms"<<endl
#ifndef TIMING
#define Debug  if(0)
#else
#define Debug 
#endif /*TIMING*/
#else
#define debug_out(X)
#define Debug if(0)
#endif /*QUIET*/

#include <stdio.h>
#include <iostream>
#include <algorithm>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <pthread.h>


//#include "aviplay_impl.h"
#include "audio.h"
#include <aviutil.h>

#include <except.h>

#define __MODULE__ "File_AudioRenderer"
using namespace std;
//template<class T> inline T min(const T x, const T y){if(x<y)return x;else return y;}
//template<class T> inline T max(const T x, const T y){if(x>y)return x;else return y;}
/**********************************************
	
	Audio queue implementation

**********************************************/
audio_queue::audio_queue()
{
    pthread_mutex_init(&mutex, NULL);
    pthread_cond_init(&cond, NULL);
    frame_in=frame_out=frame_size=0;
}
audio_queue::~audio_queue()
{
    pthread_cond_destroy(&cond);
    pthread_mutex_destroy(&mutex);
}

int audio_queue::push(const char* data, int size)
{
    pthread_mutex_lock(&mutex);
    {
	int new_pos = frame_in+size;
	if(frame_in+size>sizeof(audio_frame))size=sizeof(audio_frame)-frame_in;
	memcpy(audio_frame+frame_in, data, size);
	while(new_pos > frame_max) 
        {
	    memcpy(audio_frame, audio_frame + frame_max, new_pos - frame_max);
	    new_pos  = new_pos - frame_max;
	}	
        frame_in = new_pos;
	frame_size += size;
    }
    pthread_mutex_unlock(&mutex);
    broadcast();
    return 0;
}
int audio_queue::read(char* data, int size, int volume)
{
    pthread_mutex_lock(&mutex);
    {
        while(size>0)
	{
	    if(frame_out==frame_max)frame_out=0;
	    int step=min(size, frame_max-frame_out);
#ifdef USE_SDL
	    SDL_MixAudio((unsigned char*)data, (unsigned char*)(audio_frame+frame_out), step, volume);
#else
	    memcpy(data, audio_frame+frame_out, step);	
#endif
	    data+=step;
	    frame_out+=step;
    	    frame_size-=step;
	    size-=step;
        }	
	if(frame_size<0)
	    frame_size=0;
    }
    pthread_mutex_unlock(&mutex);
    broadcast();
    return 0;
}
int audio_queue::unread(int size)
{
    frame_out-=size;
    while(frame_out<0)frame_out+=frame_max;
    frame_size=(frame_out<=frame_in)?(frame_in-frame_out):(frame_max+frame_in-frame_out);
    return 0;
}

int audio_queue::write(int fd)//writes some data
{
    int startpos, size, tmp;
#ifdef __FreeBSD__
    const int AUDIO_BUFFER_SIZE=2048;
#else
    const int AUDIO_BUFFER_SIZE=8192;
#endif
    pthread_mutex_lock(&mutex);
    {
        size = AUDIO_BUFFER_SIZE;
	if((frame_in > frame_out) && (frame_in - frame_out < AUDIO_BUFFER_SIZE)) size = frame_in - frame_out;
        if((frame_in < frame_out) && (frame_max - frame_out < AUDIO_BUFFER_SIZE)) size = frame_max - frame_out;
    
	tmp=frame_in-frame_out;
        while(tmp<0)tmp+=frame_max;
	if(tmp!=frame_size)
        {
    	    printf("OOPS: frame_size!=tmp(%d,%d,%d)\n", frame_out, frame_in, frame_size);
        }    
	startpos=frame_out;
        if(frame_out == frame_max) frame_out = 0;
    }
    pthread_mutex_unlock(&mutex);

    int answer=::write(fd, audio_frame+startpos, size);
    if(answer==0)return 0;//sound card doesn't accept any more data ( e.g. DMA buffer is full )
    pthread_mutex_lock(&mutex);
    {
	if(frame_size==0)
	    //someone has cleared buffers while we were writing
	    //forget what we wrote, user should reset device
	{
	    pthread_mutex_unlock(&mutex);
	    return -1;
	}    
        frame_out += answer;
	if(frame_size<answer)
	    printf("OOPS: wrote %d bytes, buffer size %d\n", answer, frame_size);
	frame_size -= answer;
    }
    pthread_mutex_unlock(&mutex);
    return answer;
}
void audio_queue::clear()
{
    pthread_mutex_lock(&mutex);
    {
	frame_in=frame_out=frame_size=0;
    }	
    pthread_mutex_unlock(&mutex);
    return;
}
/*********************************************************
    
	Implementation of base audio renderer class

*********************************************************/

const char* IAudioRenderer::GetAudioFormat() const 
{
    return audio_desc;
}
IAudioRenderer::~IAudioRenderer()
{
}
void IAudioRenderer::pause(bool state)
{
    pthread_mutex_lock(&mutex);
    if(!initialized)
    {
	pthread_mutex_unlock(&mutex);
	return;
    }    
    Pause(state);
    paused=state;
    if(!state)time_start=0;
    pthread_mutex_unlock(&mutex);
}
int IAudioRenderer::Eof()
{
    if(audiostream==0)return 0;
    return audiostream->Eof();
}
void IAudioRenderer::setAsync(float async)
{
    m_async=async;
}
void IAudioRenderer::setVolume(float volume)
{
    if(volume<0)return;
    if(volume>1)return;
#ifdef USE_SDL
    m_pVolume=(int)(SDL_MIX_MAXVOLUME*volume);
#endif
}	    
void IAudioRenderer::Pause(int)
{
}
void IAudioRenderer::Reset()
{
}
double IAudioRenderer::getTime()
{
	pthread_mutex_lock(&mutex);
	if(time_start==0)
	{
	    audio_time=longcount();
	    time_start=longcount();
	    audio_realpos=audiostream->GetTime()-double(queue.size())/owf.nChannels/(owf.wBitsPerSample/8)/owf.nSamplesPerSec;
	//    audio_realpos=audiostream->GetTime()-double(m_spec.size+queue.size())/owf.nChannels/(owf.wBitsPerSample/8)/owf.nSamplesPerSec;
	//    cout<<"Cleared audio_time"<<endl;
	}
	long long time_current=longcount();
	double actual_time=audio_realpos+to_float(time_current, audio_time);
	if(actual_time>audiostream->GetTime())actual_time=audiostream->GetTime();
//	cout<<audio_realpos<<"+"<<to_float(time_current, audio_time)<<"="<<actual_time<<" at "<<double(time_current-time_start)/550000000.-actual_time<<endl;
	pthread_mutex_unlock(&mutex);
        return actual_time;
}


void IAudioRenderer::start()
{
    pthread_mutex_lock(&mutex);
    if(initialized==1)
    {
	Debug cout<<"Can't start(), already started"<<endl;
	pthread_mutex_unlock(&mutex);
	return;
    }	
    audio_realpos=0;
    audio_time=longcount();
    quit=0;
    if(audiostream)
        audiostream->SeekToTime(0);
    queue.clear();
    Pause(0);
    time_start=0;
    hangup=0;
    initialized=1;
    pthread_mutex_unlock(&mutex);
}

void IAudioRenderer::doAudioSkip(double video_time)
{
    audiostream->SeekToTime(video_time);    
}
void IAudioRenderer::doAudioExtract(double video_time)
{
    int t1, t2;
    int frames_written=0;
    
    const int one_frame_sound=max(20000, audiostream->GetFrameSize());
    const int sound_size_max=176400;

    if(quit)
	return;

    if(queue.size()>sound_size_max)
        return;

    if(audiostream->Eof())
    {
	cout<<"Audio stream finished"<<endl;
	Pause(1);
	initialized=0;
	paused=0;
	queue.broadcast();
	return;
    }	

    Unsigned ocnt;
    Unsigned samples;
    char local_frame[one_frame_sound];
    pthread_mutex_lock(&mutex);
    audiostream->ReadFrames(local_frame, one_frame_sound,
         one_frame_sound, samples, ocnt);
    Debug cout<<"doAudioExtract: Read "<<samples<<" samples ( "<<ocnt<<" bytes )"<<endl;
    /* mp3 wrong frames? */
   if (samples <= one_frame_sound)
      queue.push(local_frame, ocnt);
    pthread_mutex_unlock(&mutex);
    return;
}
void IAudioRenderer::stop()
{
    pthread_mutex_lock(&mutex);
    initialized=0;
    queue.broadcast();
//    cout<<"S T O P "<<endl;
    Pause(1);
    time_start=0;
    pthread_mutex_unlock(&mutex);
}
void IAudioRenderer::clear()
{
    queue.clear();
}
void IAudioRenderer::wake()
{
    queue.broadcast();
}
void IAudioRenderer::reseek(double pos)
{
    pthread_mutex_lock(&mutex);
    if(audiostream)
        audiostream->SeekToTime(pos);
    if(initialized==0)
    {
        pthread_mutex_unlock(&mutex);
	return;
    }
    hangup=1;
    wake();
    while(hangup)usleep(10000);
    queue.clear();
    char fake[16];
    memset(fake, 0, 16);
    queue.push(fake, 16);
    Reset();
    time_start=0;
    initialized=1;
    pthread_mutex_unlock(&mutex);
}
double IAudioRenderer::getLength()
{
    if(!audiostream)return 0;
     return audiostream->GetEndTime();
}



/*********************************************************
    
	Implementation of file output renderer class

*********************************************************/

File_AudioRenderer::File_AudioRenderer(IAviReadStream* as, int audio_fd_new)
{
    try
    {
	audiostream=as;

        if(audiostream==0)
	    throw FATAL("NULL audiostream");
	int audio_status=audiostream->StartStreaming();
	if(audio_status!=0)
	    throw FATAL("Failed to start streaming");
	int tmp;
	audiostream->GetOutputFormat(&owf, sizeof owf);
	sprintf(audio_desc, " %dkbit/s %dHz %s", (int)owf.nAvgBytesPerSec/128, (int)owf.nSamplesPerSec, 
		owf.nChannels!=1?"stereo":"mono"); 
#ifndef QUIET
	cout << "Audio format " << owf.nSamplesPerSec 
	<< "/" << owf.wBitsPerSample
	<< "/" << ((owf.nChannels!=1)?"stereo":"mono")
	<< endl;
#endif	
	audio_fd=audio_fd_new;
	try
	{
//    	    audio_fd=open("/dev/dsp",O_RDWR|O_NDELAY);
	    if(audio_fd<=0)
  		throw FATAL("Output file not open");
//	    if(audio_fd>0)
//	    {
//  		int flag;
//	        if((flag=fcntl(audio_fd,F_GETFL,0))<0)
//  	    	    throw FATAL("fcntl");
//		flag&=~O_NDELAY;
//        	if(fcntl(audio_fd,F_SETFL,flag)<0)
//		    throw FATAL("fcntl");
//	    }
	}
	catch(FatalError&)
	{
	    if(audio_fd>0)::close(audio_fd);
	    fprintf(stderr,"fatal error ocurred, exiting\n");
	    exit(1);
	}
//        if(audio_fd<=0)
//     	    throw FATAL("Can't open audio device");
/*	ioctl(audio_fd, SNDCTL_DSP_RESET, NULL);
	audio_buf_info zz;
	ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &zz);
	Snd_Limit=zz.bytes;
	cout<<"Snd_Limit "<<Snd_Limit<<endl;
	tmp=owf.nChannels-1;
	if(ioctl(audio_fd, SNDCTL_DSP_STEREO, &tmp)!=0)
	    throw FATAL("ioctl(stereo)");	    
	tmp=owf.wBitsPerSample;
	if(ioctl(audio_fd, SNDCTL_DSP_SAMPLESIZE, &tmp)!=0)
	    throw FATAL("ioctl(samplesize)");
	tmp=owf.nSamplesPerSec;
	switch(tmp)
	{
	    case 8000:
	    case 16000:
	    case 24000:
	    case 32000:
    	    case 48000:
	    
	    case 11025:
	    case 22050:
	    case 44100:
	    break;
	    default:
	    throw FATAL("Unsupported frequency");
	}    

	if(ioctl(audio_fd, SNDCTL_DSP_SPEED, &tmp)!=0)
	    throw FATAL("ioctl(speed)");
	cout<<"Format OK"<<endl;
*/
	pthread_mutex_init(&mutex, 0);
    }
    catch(FatalError& error)
    {
	if(audio_fd>0)
	    ::close(audio_fd);
	audio_fd=-1;
	audiostream=0;
	throw;
    }	
    initialized=0;
    hangup=0;
    paused=0;
    audio_realpos=0;
    audio_time=longcount();
    time_start=0;	
    quit=0;
    audiostream->SeekToTime(0);
    pthread_create(&audio_thread, NULL, doAudioOut, (void*)this);
    return;
}
void File_AudioRenderer::Reset()
{
//    if(audio_fd)
//	ioctl(audio_fd, SNDCTL_DSP_RESET, NULL);
}
File_AudioRenderer::~File_AudioRenderer()
{
    cout<<"Destroying audio renderer"<<endl;
    quit=1;
    queue.broadcast();
    pthread_join(audio_thread, NULL);
    ::close(audio_fd);
    pthread_mutex_destroy(&mutex);
    cout<<"Destroy() successful"<<endl;
}

double File_AudioRenderer::buffer_time()
{
    if(!initialized)return 0;
//    audio_buf_info zz;
//    ioctl(audio_fd, SNDCTL_DSP_GETOSPACE, &zz);
    
    double frame_time=double(queue.size()+(signed)Snd_Limit/*-zz.bytes*/)/owf.nChannels/(owf.wBitsPerSample/8)/owf.nSamplesPerSec;
    return frame_time;
}

void* File_AudioRenderer::doAudioOut(void* arg)
{
    File_AudioRenderer& a=*(File_AudioRenderer*)arg;

    int Snd_Limit=a.Snd_Limit;
    while(1)
    {
	if(a.quit)
	{
	    Debug cout<<"Exiting audio thread"<<endl;
	    return 0;
	}    
	if(a.hangup)
	{
	    a.initialized=0;
	    a.hangup=0;
	}
	if(a.paused)
	{
	    usleep(100000);
//	    cout<<"paused"<<endl;
	    continue;
	}    
	static int reset_dev=0;
	if(!a.initialized)
	{	
	    if(reset_dev)
	    {
		a.queue.clear();
//		ioctl(a.audio_fd, SNDCTL_DSP_RESET, 0);
//		reset_dev=0;
	    }
//	    usleep(100000);
//	    cout<<"Not initialized"<<endl;
	    continue;	    	    
	}else reset_dev=1;	    
	if(a.queue.wait()!=0)
	{
	    //wait() was interrupted, maybe by signal or quit=1
	    Debug cout<<"Bad wait"<<endl;
	    continue;
	}    
//	audio_buf_info zz;
//	ioctl(a.audio_fd, SNDCTL_DSP_GETOSPACE, &zz);
//	if(zz.bytes==0)
//	{
//	    usleep(10000);
//	    Debug cout<<"Buffer full"<<endl;
//	    continue;
//	}    
//	pthread_mutex_lock(&a.mutex);
/*	cout<<">"<<a.audiostream->GetTime()<<" "<<a.buffer_time()
		<<" "<<" -> "<<a.audiostream->GetTime()-a.buffer_time()
	        <<" "<<longcount()/550000000.<<endl;*/
	float in=a.getTime();
	long long ts=longcount();
//	cout<<"Before: "<<in<<endl;
//	pthread_mutex_unlock(&a.mutex);		
	int result=a.queue.write(a.audio_fd);
//	cout<<"Wrote "<<result<<" bytes"<<endl;
	if(result==0)
/*	    usleep(10000)*/;
	else
	if(result<0)
	    continue;
	else
	{
	    pthread_mutex_lock(&a.mutex);
    	    a.audio_time=longcount();
	    double frame_time=a.buffer_time();
    	    a.audio_realpos=a.audiostream->GetTime()-frame_time;
//	    cout<<a.audiostream->GetTime()<<" "<<frame_time
//		<<" "<<" -> "<<a.audio_realpos
//	        <<" "<<longcount()/550000000.<<endl;
	    pthread_mutex_unlock(&a.mutex);
	    float out=a.getTime();
	    long long tx=longcount();
//	    cout<<"Write mismatch: "<<1000.*(out-in)-to_float(tx,ts)
//	    <<" ms ( writing "<<to_float(tx,ts)<<" ms, stream difference "<<1000.*(out-in)<<" ms )"<<endl;
	}    
    }
    return 0;
}

